<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Users;

use App\Observers\Users\UserObserver;
use Carbon\Carbon;
use Database\Factories\Users\UserFactory;
use Filament\Panel;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

//Models
use App\Models\Users\Attendance;
use App\Models\Businesses\Business;
use App\Models\Users\ChangeRequest;
use App\Models\Users\Clocking;
use App\Models\Locations\Franchise;
use App\Models\Invoices\Invoice;
use App\Models\Contacts\Lead;
use App\Models\Users\LeaveRequest;
use App\Models\Users\Note;
use App\Models\Contacts\Prospect;
use App\Models\Quotes\Quote;
use App\Models\Users\Upload;
use App\Models\Users\UserDisciplinary;
use App\Models\Users\UserSetting;
use Filament\Models\Contracts\FilamentUser;
use Filament\Models\Contracts\HasName;
use App\Models\Users\FranchiseUser;

//Enums

//Traits
use App\Traits\Contacts\BelongsToContact;
use App\Traits\Users\Changable;
use App\Traits\Finances\HasBankAccount;
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;
use App\Traits\Users\Uploadable;

/**
 * Class User
 * 
 * @property int $user_id
 * @property int $contact_id
 * @property int|null $bank_account_id
 * @property Carbon|null $user_last_active_date
 * @property bool $user_can_access
 * @property string $username
 * @property string $password
 * @property Carbon $user_employment_date
 * @property int $user_leave_days
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * 
 * @property \Illuminate\Database\Eloquent\Collection|Attendance[] $attendances
 * @property \Illuminate\Database\Eloquent\Collection|Business[] $businesses
 * @property \Illuminate\Database\Eloquent\Collection|ChangeRequest[] $changeRequest
 * @property \Illuminate\Database\Eloquent\Collection|Clocking[] $clockings
 * @property \Illuminate\Database\Eloquent\Collection|Franchise[] $franchises
 * @property \Illuminate\Database\Eloquent\Collection|Invoice[] $invoices
 * @property \Illuminate\Database\Eloquent\Collection|Lead[] $leads
 * @property \Illuminate\Database\Eloquent\Collection|LeaveRequest[] $leaveRequests
 * @property \Illuminate\Database\Eloquent\Collection|Note[] $notes
 * @property \Illuminate\Database\Eloquent\Collection|Prospect[] $prospects
 * @property \Illuminate\Database\Eloquent\Collection|Quote[] $quotes
 * @property \Illuminate\Database\Eloquent\Collection|Upload[] $uploads
 * @property \Illuminate\Database\Eloquent\Collection|UserDisciplinary[] $userDisciplinaries
 * @property UserSetting|null $userSettings
 *
 * @package App\Models
 */

 #[ObservedBy(UserObserver::class)]
class User extends Authenticatable implements FilamentUser, HasName
{
	use HasBankAccount;
	use BelongsToContact;
	use Changable;
	use Uploadable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;

	const USER_ID 						= 'user_id';
	const CONTACT_ID 					= 'contact_id';
	const BANK_ACCOUNT_ID 				= 'bank_account_id';
	const USER_LAST_ACTIVE_DATE 		= 'user_last_active_date';
	const USER_CAN_ACCESS 				= 'user_can_access';
	const USERNAME 						= 'username';
	const PASSWORD 						= 'password';
	const USER_EMPLOYMENT_DATE 			= 'user_employment_date';
	const USER_LEAVE_DAYS 				= 'user_leave_days';
	const CREATED_AT 					= 'created_at';
	const UPDATED_AT 					= 'updated_at';
	const TABLE_NAME 					= 'users';
	protected $table 					= self::TABLE_NAME;
	protected $primaryKey 				= self::USER_ID;
	public $incrementing 				= true;

	protected $casts = [
		self::USER_ID 					=> 'int',
		self::CONTACT_ID 				=> 'int',
		self::BANK_ACCOUNT_ID 			=> 'int',
		self::USER_LAST_ACTIVE_DATE 	=> 'datetime',
		self::USER_CAN_ACCESS 			=> 'bool',
		self::USER_EMPLOYMENT_DATE 		=> 'datetime',
		self::USER_LEAVE_DAYS 			=> 'int',
		self::CREATED_AT 				=> 'datetime',
		self::UPDATED_AT 				=> 'datetime'
	];

	protected $hidden = [
		self::PASSWORD
	];

	protected $fillable = [
		self::CONTACT_ID,
		self::BANK_ACCOUNT_ID,
		self::USER_LAST_ACTIVE_DATE,
		self::USER_CAN_ACCESS,
		self::USERNAME,
		self::PASSWORD,
		self::USER_EMPLOYMENT_DATE,
		self::USER_LEAVE_DAYS
	];

	protected $attributes = [
		self::USER_CAN_ACCESS			=> true,
	];

	protected static array $lowercase = [
		self::USERNAME,	
	];

	/**
     * Returns filament name
     * @return string
     */
    public function getFilamentName(): string
    {
        return "{$this->username}";
    }

	public function getAuthIdentifierName(): string
    {
        return 'username';
    }

	/**
     * Returns if user can access panel
     * @param \Filament\Panel $panel
     * @return bool
     */
    public function canAccessPanel(Panel $panel): bool {

        /**
         * Return true for now
         * Test against domain later
         * However all users of the system should have access
         * Clients and Officials should have temp access - unless they make use of app interface then they should access
         * Only allow users to access once their email has been verified
         */
        //return str_ends_with($this->email, '@yourdomain.com') && $this->hasVerifiedEmail();
        return true;
    }

	/**
	 * Return the attendances for this user
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function attendances(): HasMany
	{
		return $this->hasMany(Attendance::class, self::USER_ID);
	}

	/**
	 * Returns the businesses this user has - their clients
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function businesses(): HasMany
	{
		return $this->hasMany(Business::class, self::USER_ID);
	}

	/**
	 * Return ths change requests this user has made
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function changeRequest(): HasMany
	{
		return $this->hasMany(ChangeRequest::class, self::USER_ID);
	}

	/**
	 * Return the clockings this user has made
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function clockings(): HasMany
	{
		return $this->hasMany(Clocking::class, self::USER_ID);
	}

	/**
	 * Return ths franchises this user belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function franchises(): BelongsToMany
	{
		return $this->belongsToMany(Franchise::class, 
									FranchiseUser::TABLE_NAME, 
									self::USER_ID, 
									Franchise::FRANCHISE_ID)
					->withPivot(FranchiseUser::FRANCHISE_USER_ROLE);
	}

	/**
	 * Return the invoices this user has created
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function invoices(): HasMany
	{
		return $this->hasMany(Invoice::class, self::USER_ID);
	}

	/**
	 * Return the leads this user has created
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function leads(): HasMany
	{
		return $this->hasMany(Lead::class, self::USER_ID);
	}

	/**
	 * Return the leave this user has requested
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function leaveRequests(): HasMany
	{
		return $this->hasMany(LeaveRequest::class, self::USER_ID);
	}

	/**
	 * Return the notes this user has created
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function notes(): HasMany
	{
		return $this->hasMany(Note::class, self::USER_ID);
	}

	/**
	 * Return the prospects this user has created
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function prospects(): HasMany
	{
		return $this->hasMany(Prospect::class, self::USER_ID);
	}

	/**
	 * Return the quotes this user has created
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function quotes(): HasMany
	{
		return $this->hasMany(Quote::class, self::USER_ID);
	}

	/**
	 * Return the uploads the user has made
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function uploads(): HasMany
	{
		return $this->hasMany(Upload::class, self::USER_ID);
	}

	/**
	 * Return the disciplinaries the user has against them
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function userDisciplinaries(): HasMany
	{
		return $this->hasMany(UserDisciplinary::class, self::USER_ID);
	}

	/**
	 * Return the users settings
	 * @return \Illuminate\Database\Eloquent\Relations\HasOne
	 */
	public function userSettings(): HasOne
	{
		return $this->hasOne(UserSetting::class, self::USER_ID);
	}

	public static function factory(): UserFactory
	{
		return UserFactory::new();
	}
}
