<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Users;

use App\Observers\Users\OfficialObserver;
use Carbon\Carbon;
use Database\Factories\Users\OfficialFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Users\Officialable;
use App\Models\Locations\Department;
use App\Models\Locations\Municipality;
use App\Models\Locations\LiquorAuthority;

//Enums

//Traits
use App\Traits\Contacts\BelongsToContact;
use App\Traits\Users\CanAccessTemporarily;
use App\Traits\Users\Notable;
use App\Traits\Users\Uploadable;

/**
 * Class Official
 * 
 * @property int $official_id
 * @property int $contact_id
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * 
 * @property Collection|Officialable[] $officialables
 * @property Collection|Department[] $departments
 * @property Collection|Municipality[] $municipalities
 * @property Collection|LiquorAuthority[] $liquorAuthorities
 *
 * @package App\Models
 */

#[ObservedBy(OfficialObserver::class)]
class Official extends Model
{
	use BelongsToContact;
	use CanAccessTemporarily;
	use Uploadable;
	use Notable;
	use HasFactory;

	const OFFICIAL_ID 		= 'official_id';
	const CONTACT_ID 		= 'contact_id';
	const CREATED_AT 		= 'created_at';
	const UPDATED_AT 		= 'updated_at';
	const TABLE_NAME 		= 'officials';
	protected $table 		= self::TABLE_NAME;
	protected $primaryKey 	= self::OFFICIAL_ID;
	public $incrementing 	= true;

	protected $casts = [
		self::OFFICIAL_ID 	=> 'int',
		self::CONTACT_ID 	=> 'int',
		self::CREATED_AT 	=> 'datetime',
		self::UPDATED_AT 	=> 'datetime'
	];

	protected $fillable = [
		self::CONTACT_ID
	];

	/**
	 * Returns the officialables for this official
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function officialables(): HasMany
	{
		return $this->hasMany(Officialable::class, self::OFFICIAL_ID);
	}

	/**
	 * Returns the department this official belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\MorphToMany
	 */
	public function departments()
	{
		return $this->morphedByMany(Department::class, 
									Officialable::MORPH_NAME,
									Officialable::TABLE_NAME,
									self::OFFICIAL_ID,
									Department::DEPARTMENT_ID)
					->withPivot(Department::DEPARTMENT_NAME);
	}

	/**
	 * Returns the municipality this official belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\MorphToMany
	 */
	public function municipalities()
	{
		return $this->morphedByMany(Municipality::class, 
									Officialable::MORPH_NAME,
									Officialable::TABLE_NAME,
									self::OFFICIAL_ID,
									Municipality::MUNICIPALITY_ID)
					->withPivot(Municipality::MUNICIPALITY_NAME, Municipality::MUNICIPALITY_TYPE);
	}

	/**
	 * Returns the liquor authority this official belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\MorphToMany
	 */
	public function liquorAuthorities()
	{
		return $this->morphedByMany(LiquorAuthority::class, 
									Officialable::MORPH_NAME,
									Officialable::TABLE_NAME,
									self::OFFICIAL_ID,
									LiquorAuthority::LIQUOR_AUTHORITY_ID)
					->withPivot(LiquorAuthority::LIQUOR_AUTHORITY_NAME);
	}

	public static function factory(): OfficialFactory
	{
		return OfficialFactory::new();
	}
}
