<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Users;

use App\Observers\Users\ChangeRequestObserver;
use Carbon\Carbon;
use Database\Factories\Users\ChangeRequestFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

//Models
use App\Models\Users\User;
use App\Models\Businesses\Business;
use App\Models\Users\Client;
use App\Models\Locations\Franchise;
use App\Models\Locations\Municipality;
use App\Models\Locations\LiquorAuthority;
use App\Models\Locations\Department;

//Enums
use App\Enums\Users\ChangeRequestAction;
use App\Enums\Users\ChangeRequestStatus;
use App\Enums\Users\ChangeRequestTargetType;
use App\Enums\Users\ChangeRequestType;

//Traits
use App\Traits\Users\BelongsToUser;
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;

/**
 * Class ChangeRequest
 * 
 * @property int $change_id
 * @property int $user_id
 * @property ChangeRequestType::class $changable_type
 * @property int $changable_id
 * @property ChangeRequestTargetType::class $targetable_type
 * @property int $targetable_id
 * @property string $change_reason
 * @property ChangeRequestStatus::class $change_status
 * @property ChangeRequestAction $change_action
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 *
 * @property User|null $changableUser
 * @property User|null $targetableUser
 *
 * @property Business|null $changableBusiness
 * @property Business|null $targetableBusiness
 *
 * @property Client|null $changableClient
 * @property Client|null $targetableClient
 *
 * @property Franchise|null $targetableFranchise
 * @property Municipality|null $targetableMunicipality
 * @property LiquorAuthority|null $targetableLiquorAuthority
 * @property Department|null $targetableDepartment
 * 
 * @package App\Models
 */

#[ObservedBy(ChangeRequestObserver::class)]
class ChangeRequest extends Model
{
	use BelongsToUser;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;

	const CHANGE_ID 			= 'change_id';
	const USER_ID 				= 'user_id';
	const CHANGABLE_TYPE 		= 'changable_type';
	const CHANGABLE_ID 			= 'changable_id';
	const TARGETABLE_ID 		= 'targetable_id';
	const TARGETABLE_TYPE 		= 'targetable_type';
	const CHANGE_REASON 		= 'change_reason';
	const CHANGE_STATUS 		= 'change_status';
	const CHANGE_ACTION 		= 'change_action';
	const CREATED_AT 			= 'created_at';
	const UPDATED_AT 			= 'updated_at';
	const TABLE_NAME 			= 'change_requests';
	const MORPH_NAME 			= 'changable';
	protected $table 			= self::TABLE_NAME;
	protected $primaryKey 		= 'change_id';
	public $incrementing 		= true;

	protected $casts = [
		self::CHANGE_ID 		=> 'int',
		self::USER_ID 			=> 'int',
		self::CHANGABLE_ID 		=> 'int',
		self::CHANGABLE_TYPE 	=> ChangeRequestType::class, //casts to the type that will be changed 
		self::TARGETABLE_TYPE 	=> ChangeRequestTargetType::class, //casts to the target that will be effected
		self::CHANGE_STATUS 	=> ChangeRequestStatus::class, //casts to the status set by the approver
		self::CHANGE_ACTION 	=> ChangeRequestAction::class, //casts to the action the user is requesting
		self::CREATED_AT 		=> 'datetime',
		self::UPDATED_AT 		=> 'datetime'
	];

	protected $fillable = [
		self::USER_ID,
		self::CHANGABLE_TYPE,
		self::CHANGABLE_ID,
		self::CHANGE_REASON,
		self::CHANGE_STATUS,
		self::TARGETABLE_ID,
		self::TARGETABLE_TYPE,
		self::CHANGE_ACTION
	];

	protected static array $lowercase = [
		self::CHANGE_REASON,	
	];

	/**
	 * Returns the morph entity
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo
	 */
	public function changable()
	{
		return $this->morphTo();
	}

	/**
	 * Return if the user is the one to be changed
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function changableUser()
	{
		return $this->changable_type === ChangeRequestType::USERS
			? $this->changable()
			: null;
	}

	/**
	 * Returns if the user model is the one being targeted
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function targetableUser()
	{
		return $this->targetable_type === ChangeRequestTargetType::USER
			? $this->changable()
			: null;
	}

	/**
	 * Returns if the business model is the one being targeted
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function targetableBusiness(): ?\Illuminate\Database\Eloquent\Relations\MorphTo
	{
		return $this->targetable_type === ChangeRequestTargetType::BUSINESS
			? $this->changable()
			: null;
	}

	/**
	 * Returns if the client model is the one being targeted
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function targetableClient(): ?\Illuminate\Database\Eloquent\Relations\MorphTo
	{
		return $this->targetable_type === ChangeRequestTargetType::CLIENT
			? $this->changable()
			: null;
	}

	/**
	 * Return if the client is the one to be changed
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function changableClient(): ?\Illuminate\Database\Eloquent\Relations\MorphTo
	{
		return $this->changable_type === ChangeRequestType::CLIENT
			? $this->changable()
			: null;
	}

	/**
	 * Returns if the franchise model is the one being targeted
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function targetableFranchise(): ?\Illuminate\Database\Eloquent\Relations\MorphTo
	{
		return $this->targetable_type === ChangeRequestTargetType::FRANCHISE
			? $this->changable()
			: null;
	}

	/**
	 * Returns if the municipality model is the one being targeted
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function targetableMunicipality(): ?\Illuminate\Database\Eloquent\Relations\MorphTo
	{
		return $this->targetable_type === ChangeRequestTargetType::MUNICIPALITY
			? $this->changable()
			: null;
	}

	/**
	 * Returns if the liquor authority model is the one being targeted
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function targetableLiquorAuthority(): ?\Illuminate\Database\Eloquent\Relations\MorphTo
	{
		return $this->targetable_type === ChangeRequestTargetType::LIQUOR_AUTHORIY
			? $this->changable()
			: null;
	}

	public static function factory(): ChangeRequestFactory
	{
		return ChangeRequestFactory::new();
	}
}
