<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Quotes;

use App\Observers\Quotes\QuoteItemObserver;
use Carbon\Carbon;
use Database\Factories\Quotes\QuoteItemFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Locations\ProvinceService;
use App\Models\Quotes\Quote;
use App\Models\Quotes\QuoteItemExpense;

//Enums

//Traits
use App\Traits\Items\BelongsToItem;
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;

/**
 * Class QuoteItem
 * 
 * @property int $quote_item_id
 * @property int $quote_id
 * @property int|null $item_id
 * @property int|null $province_service_id
 * @property Carbon $item_date_added
 * @property string $item_name
 * @property string $item_description
 * @property float $item_quantity
 * @property float $item_professional_fee
 * @property float $item_miscellaneous_fee
 * @property float $item_price
 * @property float $item_discount
 * @property float $item_tax_total
 * @property float $item_subtotal
 * @property float $item_expense_total
 * @property float $item_total
 * 
 * @property ProvinceService|null $provinceService
 * @property Quote $quote
 * @property Collection|QuoteItemExpense[] $expenses
 *
 * @package App\Models
 */

 #[ObservedBy(QuoteItemObserver::class)]
class QuoteItem extends Model
{
	use BelongsToItem;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const QUOTE_ITEM_ID 				= 'quote_item_id';
	const QUOTE_ID 						= 'quote_id';
	const ITEM_ID 						= 'item_id';
	const PROVINCE_SERVICE_ID 			= 'province_service_id';
	const ITEM_DATE_ADDED 				= 'item_date_added';
	const ITEM_NAME 					= 'item_name';
	const ITEM_DESCRIPTION 				= 'item_description';
	const ITEM_QUANTITY 				= 'item_quantity';
	const ITEM_PROFESSIONAL_FEE 		= 'item_professional_fee';
	const ITEM_MISCELLANEOUS_FEE 		= 'item_miscellaneous_fee';
	const ITEM_PRICE 					= 'item_price';
	const ITEM_DISCOUNT 				= 'item_discount';
	const ITEM_TAX_TOTAL 				= 'item_tax_total';
	const ITEM_SUBTOTAL 				= 'item_subtotal';
	const ITEM_EXPENSE_TOTAL 			= 'item_expense_total';
	const ITEM_TOTAL 					= 'item_total';
	const TABLE_NAME 					= 'quote_items';
	protected $table 					= self::TABLE_NAME;
	protected $primaryKey 				= self::QUOTE_ITEM_ID;
	public $incrementing 				= true;
	public $timestamps 					= false;

	protected $casts = [
		self::QUOTE_ITEM_ID 			=> 'int',
		self::QUOTE_ID 					=> 'int',
		self::ITEM_ID 					=> 'int',
		self::PROVINCE_SERVICE_ID 		=> 'int',
		self::ITEM_DATE_ADDED 			=> 'datetime',
		self::ITEM_QUANTITY 			=> 'float',
		self::ITEM_PROFESSIONAL_FEE		=> 'float',
		self::ITEM_MISCELLANEOUS_FEE	=> 'float',
		self::ITEM_PRICE 				=> 'float',
		self::ITEM_DISCOUNT 			=> 'float',
		self::ITEM_TAX_TOTAL 			=> 'float',
		self::ITEM_SUBTOTAL 			=> 'float',
		self::ITEM_EXPENSE_TOTAL 		=> 'float',
		self::ITEM_TOTAL 				=> 'float'
	];

	protected $fillable = [
		self::QUOTE_ID,
		self::ITEM_ID,
		self::PROVINCE_SERVICE_ID,
		self::ITEM_DATE_ADDED,
		self::ITEM_NAME,
		self::ITEM_DESCRIPTION,
		self::ITEM_QUANTITY,
		self::ITEM_PROFESSIONAL_FEE,
		self::ITEM_MISCELLANEOUS_FEE,
		self::ITEM_PRICE,
		self::ITEM_DISCOUNT,
		self::ITEM_TAX_TOTAL,
		self::ITEM_SUBTOTAL,
		self::ITEM_EXPENSE_TOTAL,
		self::ITEM_TOTAL
	];

	protected $atributes = [
		self::ITEM_MISCELLANEOUS_FEE	=> 0.0,
		self::ITEM_PROFESSIONAL_FEE		=> 0.0,
		self::ITEM_DISCOUNT				=> 0.0,
		self::ITEM_TAX_TOTAL			=> 0.0,
		self::ITEM_SUBTOTAL				=> 0.0,
		self::ITEM_EXPENSE_TOTAL		=> 0.0,
		self::ITEM_TOTAL				=> 0.0,
	];

	protected static array $lowercase = [
		self::ITEM_NAME,
		self::ITEM_DESCRIPTION,	
	];

	/**
	 * Return the province service this quote item uses
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function provinceService(): BelongsTo
	{
		return $this->belongsTo(ProvinceService::class, ProvinceService::PROVINCE_SERVICE_ID);
	}

	/**
	 * Return the quote this quote item belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function quote(): BelongsTo
	{
		return $this->belongsTo(Quote::class, Quote::QUOTE_ID);
	}

	/**
	 * Return the expense attached to this quote item
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function expenses(): HasMany
	{
		return $this->hasMany(QuoteItemExpense::class, self::QUOTE_ITEM_ID);
	}

	public static function factory(): QuoteItemFactory
	{
		return QuoteItemFactory::new();
	}
}
