<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Quotes;

use App\Observers\Quotes\QuoteObserver;
use Carbon\Carbon;
use Database\Factories\Quotes\QuoteFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

//Models
use App\Models\Invoices\Invoice;
use App\Models\Contacts\Prospect;
use App\Models\Items\Item;
use App\Models\Quotes\QuoteItem;

//Enums
use App\Enums\Quotes\QuoteStatus;

//Traits
use App\Traits\Businesses\BelongsToBusiness;
use App\Traits\Locations\BelongsToFranchise;
use App\Traits\Users\BelongsToUser;
use App\Traits\Users\Notable;
use App\Traits\Users\Uploadable;

/**
 * Class Quote
 * 
 * @property int $quote_id
 * @property int $user_id
 * @property int|null $business_id
 * @property int $franchise_id
 * @property string $quote_number
 * @property QuoteStatus::class $quote_status
 * @property string|null $quote_password
 * @property string|null $quote_url_key
 * @property float|null $quote_item_discount_total
 * @property float|null $quote_item_tax_total
 * @property float|null $quote_item_subtotal
 * @property float|null $quote_total
 * @property Carbon|null $quote_date_expires
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property string|null $deleted_at
 * 
 * @property Collection|Invoice[] $invoices
 * @property Collection|Prospect[] $prospect
 * @property Collection|Item[] $items
 * @property Collection|QuoteItem[] $quoteItems
 *
 * @package App\Models
 */

 #[ObservedBy(QuoteObserver::class)]
class Quote extends Model
{
	use SoftDeletes;
	use BelongsToUser;
	use BelongsToBusiness;
	use Uploadable;
	use Notable;
	use HasFactory;
	use BelongsToFranchise; //A quote can belong to the franchise of the user that made it, it will be the franchise that the user made it from

	const QUOTE_ID 						= 'quote_id';
	const USER_ID 						= 'user_id';
	const BUSINESS_ID 					= 'business_id';
	const FRANCHISE_ID 					= 'franchise_id';
	const QUOTE_NUMBER 					= 'quote_number';
	const QUOTE_STATUS 					= 'quote_status';
	const QUOTE_PASSWORD 				= 'quote_password';
	const QUOTE_URL_KEY 				= 'quote_url_key';
	const QUOTE_ITEM_DISCOUNT_TOTAL 	= 'quote_item_discount_total';
	const QUOTE_ITEM_TAX_TOTAL 			= 'quote_item_tax_total';
	const QUOTE_ITEM_SUBTOTAL 			= 'quote_item_subtotal';
	const QUOTE_TOTAL 					= 'quote_total';
	const QUOTE_DATE_EXPIRES 			= 'quote_date_expires';
	const CREATED_AT 					= 'created_at';
	const UPDATED_AT 					= 'updated_at';
	const DELETED_AT 					= 'deleted_at';
	const TABLE_NAME 					= 'quotes';
	protected $table 					= self::TABLE_NAME;
	protected $primaryKey 				= self::QUOTE_ID;
	public $incrementing 				= true;

	protected $casts = [
		self::QUOTE_ID 					=> 'int',
		self::USER_ID 					=> 'int',
		self::BUSINESS_ID 				=> 'int',
		self::FRANCHISE_ID 				=> 'int',
		self::QUOTE_STATUS 				=> QuoteStatus::class,
		self::QUOTE_ITEM_DISCOUNT_TOTAL => 'float',
		self::QUOTE_ITEM_TAX_TOTAL 		=> 'float',
		self::QUOTE_ITEM_SUBTOTAL 		=> 'float',
		self::QUOTE_TOTAL 				=> 'float',
		self::QUOTE_DATE_EXPIRES 		=> 'datetime',
		self::CREATED_AT 				=> 'datetime',
		self::UPDATED_AT 				=> 'datetime'
	];

	protected $hidden = [
		self::QUOTE_PASSWORD
	];

	protected $fillable = [
		self::USER_ID,
		self::BUSINESS_ID,
		self::FRANCHISE_ID,
		self::QUOTE_NUMBER,
		self::QUOTE_STATUS,
		self::QUOTE_PASSWORD,
		self::QUOTE_URL_KEY,
		self::QUOTE_ITEM_DISCOUNT_TOTAL,
		self::QUOTE_ITEM_TAX_TOTAL,
		self::QUOTE_ITEM_SUBTOTAL,
		self::QUOTE_TOTAL,
		self::QUOTE_DATE_EXPIRES
	];

	protected $attributes = [
		self::QUOTE_ITEM_DISCOUNT_TOTAL	=> 0.0,
		self::QUOTE_ITEM_TAX_TOTAL		=> 0.0,
		self::QUOTE_ITEM_SUBTOTAL		=> 0.0,
		self::QUOTE_TOTAL				=> 0.0,
	];

	/**
	 * Return the invoice(s) this quote generated
	 * - A quote can be used to generate multiple invoices when an invoice is deleted or cancelled
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function invoices(): HasMany
	{
		return $this->hasMany(Invoice::class, self::QUOTE_ID);
	}

	/**
	 * Returns the prospect attached to this quote
	 * @return \Illuminate\Database\Eloquent\Relations\HasOne
	 */
	public function prospect(): HasOne
	{
		return $this->hasOne(Prospect::class, self::QUOTE_ID);
	}

	/**
	 * returns the items that this quote uses
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function items(): BelongsToMany
	{
		return $this->belongsToMany(Item::class, 
									QuoteItem::TABLE_NAME, 
									self::QUOTE_ID, 
									Item::ITEM_ID)
					->withPivot(QuoteItem::QUOTE_ITEM_ID, QuoteItem::PROVINCE_SERVICE_ID,
								QuoteItem::ITEM_DATE_ADDED, QuoteItem::ITEM_NAME, 
								QuoteItem::ITEM_DESCRIPTION, QuoteItem::ITEM_QUANTITY, 
								QuoteItem::ITEM_PROFESSIONAL_FEE, QuoteItem::ITEM_MISCELLANEOUS_FEE, 
								QuoteItem::ITEM_PRICE, QuoteItem::ITEM_DISCOUNT, 
								QuoteItem::ITEM_TAX_TOTAL, QuoteItem::ITEM_SUBTOTAL, 
								QuoteItem::ITEM_TOTAL);
	}

	/**
	 * Returns the quote items, where as item returns the item
	 * - This would be used if the item is deleted, we soft copied the item into quote items
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function quoteItems()
	{	
		return $this->hasMany(QuoteItem::class, self::QUOTE_ID);
	}

	public static function factory(): QuoteFactory
	{
		return QuoteFactory::new();
	}
}
