<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Locations;

use Database\Factories\Locations\ProvinceServiceFactory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Invoices\InvoiceItem;
use App\Models\Items\Item;
use App\Models\Quotes\QuoteItem;

//Enums

//Traits
use App\Traits\Locations\BelongsToProvince;
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;

/**
 * Class ProvinceService
 * 
 * @property int $province_service_id
 * @property int $province_id
 * @property string $service_name
 * @property string $service_description
 *
 * @property Collection|InvoiceItem[] $invoiceItems
 * @property Collection|Item[] $items
 * @property Collection|QuoteItem[] $quoteItems
 *
 * @package App\Models
 */
class ProvinceService extends Model
{
	use BelongsToProvince;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const PROVINCE_SERVICE_ID 		= 'province_service_id';
	const PROVINCE_ID 				= 'province_id';
	const SERVICE_NAME 				= 'service_name';
	const SERVICE_DESCRIPTION 		= 'service_description';
	const TABLE_NAME 				= 'province_services';
	protected $table 				= self::TABLE_NAME;
	protected $primaryKey 			= self::PROVINCE_SERVICE_ID;
	public $timestamps 				= false;
	public $incrementing 			= true;

	protected $casts = [
		self::PROVINCE_SERVICE_ID 	=> 'int',
		self::PROVINCE_ID 			=> 'int'
	];

	protected $fillable = [
		self::PROVINCE_ID,
		self::SERVICE_NAME,
		self::SERVICE_DESCRIPTION
	];

	protected static array $lowercase = [
		self::SERVICE_NAME,
		self::SERVICE_DESCRIPTION,	
	];

	/**
	 * Returns the items attached to an invoice where this service was used
	 * - This relationship is mainly for invoices to easily make use of the services
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function invoiceItems(): HasMany
	{
		return $this->hasMany(InvoiceItem::class, self::PROVINCE_SERVICE_ID);
	}

	/**
	 * Returns the items that make use of this service
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function items(): HasMany
	{
		return $this->hasMany(Item::class, self::PROVINCE_SERVICE_ID);
	}

	/**
	 * Returns the items attached to a quote where this service was used
	 * - This relationship is mainly for quotes to easily make use of the services
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function quoteItems(): HasMany
	{
		return $this->hasMany(QuoteItem::class, self::PROVINCE_SERVICE_ID);
	}

	public static function factory(): ProvinceServiceFactory
	{
		return ProvinceServiceFactory::new();
	}
}
