<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Locations;

use Database\Factories\Locations\ProvinceFactory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Locations\District;
use App\Models\Locations\Franchise;
use App\Models\Locations\LiquorAuthority;
use App\Models\Items\Item;
use App\Models\Locations\ProvinceService;
use App\Models\Items\ProvinceItem;

//Enums
use App\Enums\Locations\ProvinceType;

//Traits
use App\Traits\Finances\HasBankAccount;
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;
use App\Traits\Users\Uploadable;

/**
 * Class Province
 * 
 * @property int $province_id
 * @property int|null $bank_account_id
 * @property string $province_name
 * @property string $province_acronym
 * 
 * @property Collection|District[] $districts
 * @property Collection|Franchise[] $franchises
 * @property Collection|LiquorAuthority[] $liquorAuthorities
 * @property Collection|Item[] $items
 * @property Collection|ProvinceService[] $provinceServices
 *
 * @package App\Models
 */
class Province extends Model
{
	use HasBankAccount;
	use Uploadable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;

	const PROVINCE_ID 			= 'province_id';
	const BANK_ACCOUNT_ID 		= 'bank_account_id';
	const PROVINCE_NAME 		= 'province_name';
	const PROVINCE_ACRONYM 		= 'province_acronym';
	const TABLE_NAME 			= 'provinces';
	protected $table 			= self::TABLE_NAME;
	protected $primaryKey 		= self::PROVINCE_ID;
	public $timestamps 			= false;
	public $incrementing 		= true;

	protected $casts = [
		self::PROVINCE_ID 		=> 'int',
		self::BANK_ACCOUNT_ID 	=> 'int',
		self::PROVINCE_NAME		=> ProvinceType::class,
	];

	protected $fillable = [
		self::BANK_ACCOUNT_ID,
		self::PROVINCE_NAME,
		self::PROVINCE_ACRONYM
	];

	protected static array $lowercase = [
		self::PROVINCE_ACRONYM,	
	];

	/**
	 * Return the districts this province has
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function districts(): HasMany
	{
		return $this->hasMany(District::class, self::PROVINCE_ID);
	}

	/**
	 * Return the franchises this province has
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function franchises(): HasMany
	{
		return $this->hasMany(Franchise::class, Self::PROVINCE_ID);
	}

	/**
	 * Return the liquor authorities this province has
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function liquorAuthorities(): HasMany
	{
		return $this->hasMany(LiquorAuthority::class, self::PROVINCE_ID);
	}

	/**
	 * Returns the items for this province
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function items(): BelongsToMany
	{
		return $this->belongsToMany(Item::class, 
									ProvinceItem::TABLE_NAME, 
									self::PROVINCE_ID, 
									Item::ITEM_ID)
					->withPivot(ProvinceItem::PROVINCE_ID, ProvinceItem::ITEM_ID, ProvinceItem::PROVINCE_ITEM_ID);
	}

	/**
	 * Returns the province items this province has
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function provinceItems() {
		return $this->hasMany(ProvinceItem::class, self::PROVINCE_ID);
	}

	/**
	 * Returns the services this province uses
	 * - A service would then be used on the quote/invoice
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function provinceServices(): HasMany
	{
		return $this->hasMany(ProvinceService::class, self::PROVINCE_ID);
	}

	public static function factory(): ProvinceFactory
	{
		return ProvinceFactory::new();
	}
}
