<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Locations;

use Database\Factories\Locations\MunicipalityFactory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Locations\District;
use App\Models\Locations\Franchise;

//Enums
use App\Enums\Locations\MunicipalityType;

//Traits
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;
use App\Traits\Users\Officialable;

/**
 * Class Municipality
 * 
 * @property int $municipality_id
 * @property int $district_id
 * @property string $municipality_name
 * @property MunicipalityType::class $municipality_type
 * @property int|null $parent_id
 * 
 * @property District $district
 * @property Municipality|null $parent
 * @property Collection|Franchise[] $franchises
 * @property Collection|Municipality[] $children
 *
 * @package App\Models
 */
class Municipality extends Model
{
	use Officialable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const MUNICIPALITY_ID 			= 'municipality_id';
	const DISTRICT_ID 				= 'district_id';
	const MUNICIPALITY_NAME 		= 'municipality_name';
	const MUNICIPALITY_CODE 		= 'municipality_code';
	const MUNICIPALITY_TYPE 		= 'municipality_type';
	const PARENT_ID 				= 'parent_id';
	const TABLE_NAME 				= 'municipalities';
	protected $table 				= self::TABLE_NAME;
	protected $primaryKey 			= self::MUNICIPALITY_ID;
	public $incrementing 			= true;
	public $timestamps 				= false;

	protected $casts = [
		self::MUNICIPALITY_ID 		=> 'int',
		self::DISTRICT_ID 			=> 'int',
		self::MUNICIPALITY_TYPE 	=> MunicipalityType::class,
		self::PARENT_ID 			=> 'int'
	];

	protected $fillable = [
		self::DISTRICT_ID,
		self::MUNICIPALITY_NAME,
		self::MUNICIPALITY_CODE,
		self::MUNICIPALITY_TYPE,
		self::PARENT_ID
	];

	protected static array $lowercase = [
		self::MUNICIPALITY_NAME,
		self::MUNICIPALITY_CODE	
	];

	/**
	 * NOTE: The parent and children relationships might overwrite the packages relationships
	 */

	/**
	 * Return the district this municipality belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function district(): BelongsTo
	{
		return $this->belongsTo(District::class, District::DISTRICT_ID);
	}

	/**
	 * Returns the parent for this municipality
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function parent(): BelongsTo
	{
		return $this->belongsTo(Municipality::class, self::PARENT_ID);
	}

	/**
	 * Returns the franchises this municipality has
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function franchises(): HasMany
	{
		return $this->hasMany(Franchise::class, self::MUNICIPALITY_ID);
	}

	/**
	 * Returns the children this municipality has
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function children(): HasMany
	{
		return $this->hasMany(Municipality::class, self::PARENT_ID);
	}

	public static function factory(): MunicipalityFactory
	{
		return MunicipalityFactory::new();
	}
}
