<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Locations;

use Database\Factories\Locations\FranchiseFactory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Locations\Municipality;
use App\Models\Locations\Province;
use App\Models\Locations\Department;
use App\Models\Users\User;
use App\Models\Users\FranchiseUser;

//Enums

//Traits
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Contacts\ManyAddressable;
use App\Traits\Users\Notable;
use App\Traits\Users\Uploadable;

/**
 * Class Franchise
 * 
 * @property int $franchise_id
 * @property int $province_id
 * @property int|null $municipality_id
 * @property string $franchise_name
 * @property string $franchise_acronym
 * @property string|null $settings_package
 * @property bool $setting_use_global
 * 
 * @property Municipality $municipality
 * @property Province $province
 * @property Collection|Department[] $departments
 * @property Collection|User[] $users
 *
 * @package App\Models
 */
class Franchise extends Model
{
	use ManyAddressable;
	use Uploadable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;

	const FRANCHISE_ID 				= 'franchise_id';
	const PROVINCE_ID 				= 'province_id';
	const MUNICIPALITY_ID 			= 'municipality_id';
	const FRANCHISE_NAME 			= 'franchise_name';
	const FRANCHISE_ACRONYM 		= 'franchise_acronym';
	const SETTINGS_PACKAGE 			= 'settings_package';
	const SETTING_USE_GLOBAL 		= 'setting_use_global';
	const NEXT_ID					='next_id';
	const TABLE_NAME 				= 'franchises';
	protected $table 				= self::TABLE_NAME;
	protected $primaryKey 			= self::FRANCHISE_ID;
	public $incrementing 			= true;
	public $timestamps 				= false;

	protected $casts = [
		self::FRANCHISE_ID 			=> 'int',
		self::PROVINCE_ID 			=> 'int',
		self::MUNICIPALITY_ID 		=> 'int',
		self::SETTINGS_PACKAGE 		=> 'array',
		self::SETTING_USE_GLOBAL 	=> 'bool',
		self::NEXT_ID				=> 'int',
	];

	protected $fillable = [
		self::PROVINCE_ID,
		self::MUNICIPALITY_ID,
		self::FRANCHISE_NAME,
		self::FRANCHISE_ACRONYM,
		self::SETTINGS_PACKAGE,
		self::SETTING_USE_GLOBAL,
		self::NEXT_ID,
	];

	protected $attributes = [
		self::SETTING_USE_GLOBAL	=> false,
		self::NEXT_ID				=> 0,
	];

	protected static array $lowercase = [
		self::FRANCHISE_ACRONYM,
		self::FRANCHISE_NAME
	];

	/**
	 * Returns the municipality this franchise belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function municipality(): BelongsTo
	{
		return $this->belongsTo(Municipality::class, Municipality::MUNICIPALITY_ID);
	}

	/**
	 * REturns the province this franchise belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function province(): BelongsTo
	{
		return $this->belongsTo(Province::class, Province::PROVINCE_ID);
	}

	/**
	 * Returns the departments this franchise has
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function departments(): HasMany
	{
		return $this->hasMany(Department::class, self::FRANCHISE_ID);
	}

	/**
	 * Returns the users of this franchise
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function users(): BelongsToMany
	{
		return $this->belongsToMany(User::class, 
									FranchiseUser::TABLE_NAME, 
									self::FRANCHISE_ID, 
									User::USER_ID)
					->withPivot(FranchiseUser::FRANCHISE_USER_ID, FranchiseUser::FRANCHISE_USER_ROLE);
	}

	public static function factory(): FranchiseFactory
	{
		return FranchiseFactory::new();
	}
}
