<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Items;

use Database\Factories\Items\ItemFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Businesses\Business;
use App\Models\Invoices\Invoice;
use App\Models\Items\ItemExpense;
use App\Models\Items\Step;
use App\Models\Locations\Province;
use App\Models\Quotes\Quote;
use App\Models\Items\Category;
use App\Models\Locations\ProvinceService;
use App\Models\Invoices\InvoiceItem;
use App\Models\Items\BusinessItem;
use App\Models\Items\ItemStep;
use App\Models\Quotes\QuoteItem;
use App\Models\Items\ProvinceItem;

//Enums

//Traits
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;
use App\Traits\Users\Uploadable;

/**
 * Class Item
 * 
 * @property int $item_id
 * @property int|null $category_id
 * @property int|null $province_service_id
 * @property string $item_name
 * @property string $item_description
 * @property float $item_price
 * 
 * @property Category|null $category
 * @property ProvinceService|null $provinceService
 * @property \Illuminate\Database\Eloquent\Collection|Business[] $businesses
 * @property \Illuminate\Database\Eloquent\Collection|Invoice[] $invoices
 * @property \Illuminate\Database\Eloquent\Collection|ItemExpense[] $itemExpenses
 * @property \Illuminate\Database\Eloquent\Collection|Step[] $steps
 * @property \Illuminate\Database\Eloquent\Collection|Province[] $provinces
 * @property \Illuminate\Database\Eloquent\Collection|Quote[] $quotes
 *
 * @package App\Models
 */
class Item extends Model
{
	use Uploadable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const ITEM_ID 					= 'item_id';
	const CATEGORY_ID 				= 'category_id';
	const PROVINCE_SERVICE_ID 		= 'province_service_id';
	const ITEM_NAME 				= 'item_name';
	const ITEM_DESCRIPTION 			= 'item_description';
	const ITEM_PRICE 				= 'item_price';
	const TABLE_NAME 				= 'items';
	protected $table 				= self::TABLE_NAME;
	protected $primaryKey 			= self::ITEM_ID;
	public $incrementing 			= true;
	public $timestamps 				= false;

	protected $casts = [
		self::ITEM_ID 				=> 'int',
		self::CATEGORY_ID 			=> 'int',
		self::PROVINCE_SERVICE_ID 	=> 'int',
		self::ITEM_PRICE 			=> 'float'
	];

	protected $fillable = [
		self::CATEGORY_ID,
		self::PROVINCE_SERVICE_ID,
		self::ITEM_NAME,
		self::ITEM_DESCRIPTION,
		self::ITEM_PRICE
	];

	protected static array $lowercase = [
		self::ITEM_NAME,
		self::ITEM_DESCRIPTION,	
	];

	/**
	 * Returns the category this item belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function category(): BelongsTo
	{
		return $this->belongsTo(Category::class, Category::CATEGORY_ID);
	}

	/**
	 * Returns the province service this item makes use of
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function provinceService(): BelongsTo
	{
		return $this->belongsTo(ProvinceService::class, ProvinceService::PROVINCE_SERVICE_ID);
	}

	/**
	 * Returns the businesses that make use of this item
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function businesses(): BelongsToMany
	{
		return $this->belongsToMany(Business::class, 
									BusinessItem::TABLE_NAME, 
									self::ITEM_ID, 
									Business::BUSINESS_ID)
					->withPivot(BusinessItem::BUSINESS_ITEM_ID, BusinessItem::INVOICE_ID, 
								BusinessItem::ITEM_STEP_ID, BusinessItem::APPROVED_DATE, 
								BusinessItem::RENEW_DATE, BusinessItem::EXPIRY_DATE, 
								BusinessItem::ITEM_STATUS, BusinessItem::ITEM_NAME, 
								BusinessItem::ITEM_DESCRIPTION, BusinessItem::ITEM_PRICE)
					->withTimestamps();
	}

	/**
	 * Returns the invoices that make use of this item
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function invoices(): BelongsToMany
	{
		return $this->belongsToMany(Invoice::class, 
									InvoiceItem::TABLE_NAME, 
									self::ITEM_ID, 
									Invoice::INVOICE_ID)
					->withPivot(InvoiceItem::INVOICE_ITEM_ID, InvoiceItem::PROVINCE_SERVICE_ID, 
								InvoiceItem::ITEM_DATE_ADDED, InvoiceItem::ITEM_NAME, 
								InvoiceItem::ITEM_DESCRIPTION, InvoiceItem::ITEM_QUANTITY, 
								InvoiceItem::ITEM_PROFESSIONAL_FEE, InvoiceItem::ITEM_MISCELLANEOUS_FEE, 
								InvoiceItem::ITEM_PRICE, InvoiceItem::ITEM_DISCOUNT, 
								InvoiceItem::ITEM_TAX_TOTAL, InvoiceItem::ITEM_SUBTOTAL, 
								InvoiceItem::ITEM_TOTAL);
	}

	/**
	 * Returns the item expense of this item
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function itemExpenses(): HasMany
	{
		return $this->hasMany(ItemExpense::class, self::ITEM_ID);
	}

	/**
	 * Returns the steps this item has
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function steps(): BelongsToMany
	{
		return $this->belongsToMany(Step::class, 
									ItemStep::TABLE_NAME, 
									self::ITEM_ID, 
									Step::STEP_ID)
					->withPivot(ItemStep::ITEM_STEP_ID);
	}

	/**
	 * Returns the provinces that make use of this item
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function provinces(): BelongsToMany
	{
		return $this->belongsToMany(Province::class, 
									ProvinceItem::TABLE_NAME, 
									self::ITEM_ID, 
									Province::PROVINCE_ID)
					->withPivot(ProvinceItem::PROVINCE_ITEM_ID);
	}

	/**
	 * Returns the quotes that use this item
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function quotes(): BelongsToMany
	{
		return $this->belongsToMany(Quote::class, 
									QuoteItem::TABLE_NAME, 
									self::ITEM_ID, 
									QUOTE::QUOTE_ID)
					->withPivot(QuoteItem::QUOTE_ITEM_ID, QuoteItem::PROVINCE_SERVICE_ID, 
								QuoteItem::ITEM_DATE_ADDED, QuoteItem::ITEM_NAME, 
								QuoteItem::ITEM_DESCRIPTION, QuoteItem::ITEM_QUANTITY, 
								QuoteItem::ITEM_PROFESSIONAL_FEE, QuoteItem::ITEM_MISCELLANEOUS_FEE, 
								QuoteItem::ITEM_PRICE, QuoteItem::ITEM_DISCOUNT, 
								QuoteItem::ITEM_TAX_TOTAL, QuoteItem::ITEM_SUBTOTAL, 
								QuoteItem::ITEM_TOTAL);
	}

	public static function factory(): ItemFactory
	{
		return ItemFactory::new();
	}
}
