<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Items;

use Carbon\Carbon;
use Database\Factories\Items\BusinessItemFactory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Invoices\Invoice;
use App\Models\Items\ItemStep;
use App\Models\Items\BusinessItemExtraField;

//Enums
use App\Enums\Items\BusinessItemStatus;

//Traits
use App\Traits\Businesses\BelongsToBusiness;
use App\Traits\Items\BelongsToItem;
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;
use App\Traits\Users\Uploadable;

/**
 * Class BusinessItem
 * 
 * @property int $business_item_id
 * @property int $business_id
 * @property int|null $item_id
 * @property int|null $invoice_id
 * @property int $item_step_id
 * @property Carbon|null $approved_date
 * @property Carbon|null $renew_date
 * @property Carbon|null $expiry_date
 * @property BusinessItemStatus::class $item_status
 * @property string $item_name
 * @property string $item_description
 * @property float $item_price
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * 
 * @property Invoice|null $invoice
 * @property ItemStep $itemStep
 * @property Collection|BusinessItemExtraField[] $extraFields
 *
 * @package App\Models
 */
class BusinessItem extends Model
{
	use BelongsToBusiness;
	use BelongsToItem;
	use Uploadable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;

	const BUSINESS_ITEM_ID 		= 'business_item_id';
	const BUSINESS_ID 			= 'business_id';
	const ITEM_ID 				= 'item_id';
	const INVOICE_ID 			= 'invoice_id';
	const ITEM_STEP_ID 			= 'item_step_id';
	const APPROVED_DATE 		= 'approved_date';
	const RENEW_DATE 			= 'renew_date';
	const EXPIRY_DATE 			= 'expiry_date';
	const ITEM_STATUS 			= 'item_status';
	const ITEM_NAME 			= 'item_name';
	const ITEM_DESCRIPTION 		= 'item_description';
	const ITEM_PRICE 			= 'item_price';
	const CREATED_AT 			= 'created_at';
	const UPDATED_AT 			= 'updated_at';
	const TABLE_NAME 			= 'business_items';
	protected $table 			= self::TABLE_NAME;
	protected $primaryKey 		= 'business_item_id';
	public $incrementing 		= true;

	protected $casts = [
		self::BUSINESS_ITEM_ID 	=> 'int',
		self::BUSINESS_ID 		=> 'int',
		self::ITEM_ID 			=> 'int',
		self::INVOICE_ID 		=> 'int',
		self::ITEM_STEP_ID 		=> 'int',
		self::APPROVED_DATE 	=> 'datetime',
		self::RENEW_DATE 		=> 'datetime',
		self::EXPIRY_DATE 		=> 'datetime',
		self::ITEM_STATUS 		=> BusinessItemStatus::class,
		self::ITEM_PRICE 		=> 'float',
		self::CREATED_AT 		=> 'datetime',
		self::UPDATED_AT 		=> 'datetime'
	];

	protected $fillable = [
		self::BUSINESS_ID,
		self::ITEM_ID,
		self::INVOICE_ID,
		self::ITEM_STEP_ID,
		self::APPROVED_DATE,
		self::RENEW_DATE,
		self::EXPIRY_DATE,
		self::ITEM_STATUS,
		self::ITEM_NAME,
		self::ITEM_DESCRIPTION,
		self::ITEM_PRICE
	];

	protected static array $lowercase = [
		self::ITEM_NAME,
		self::ITEM_DESCRIPTION,	
	];

	/**
	 * Returns the invoice associated with this business item
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function invoice(): BelongsTo
	{
		return $this->belongsTo(Invoice::class, Invoice::INVOICE_ID);
	}

	/**
	 * Returns the current item step for this business item
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function itemStep(): BelongsTo
	{
		return $this->belongsTo(ItemStep::class, ItemStep::ITEM_STEP_ID);
	}

	/**
	 * Returns the extra fields attached to this business item
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function extraFields(): HasMany
	{
		return $this->hasMany(BusinessItemExtraField::class, self::BUSINESS_ITEM_ID);
	}

	public static function factory(): BusinessItemFactory
	{
		return BusinessItemFactory::new();
	}
}
