<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Invoices;

use App\Observers\Invoices\InvoiceItemObserver;
use Carbon\Carbon;
use Database\Factories\Invoices\InvoiceItemFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Invoices\Invoice;
use App\Models\Items\Item;
use App\Models\Locations\ProvinceService;
use App\Models\Invoices\InvoiceItemExpense;

//Enums

//Traits
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;

/**
 * Class InvoiceItem
 * 
 * @property int $invoice_item_id
 * @property int $invoice_id
 * @property int|null $item_id
 * @property int|null $province_service_id
 * @property Carbon $item_date_added
 * @property string $item_name
 * @property string $item_description
 * @property float $item_quantity
 * @property float $item_professional_fee
 * @property float $item_miscellaneous_fee
 * @property float $item_price
 * @property float $item_discount
 * @property float $item_tax_total
 * @property float $item_subtotal
 * @property float $item_expense_total
 * @property float $item_total
 * 
 * @property Invoice $invoice
 * @property Item|null $item
 * @property ProvinceService|null $provinceService
 * @property Collection|InvoiceItemExpense[] $expenses
 *
 * @package App\Models
 */

#[ObservedBy(InvoiceItemObserver::class)]
class InvoiceItem extends Model
{
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const INVOICE_ITEM_ID 				= 'invoice_item_id';
	const INVOICE_ID 					= 'invoice_id';
	const ITEM_ID 						= 'item_id';
	const PROVINCE_SERVICE_ID 			= 'province_service_id';
	const ITEM_DATE_ADDED 				= 'item_date_added';
	const ITEM_NAME 					= 'item_name';
	const ITEM_DESCRIPTION 				= 'item_description';
	const ITEM_QUANTITY 				= 'item_quantity';
	const ITEM_PROFESSIONAL_FEE 		= 'item_professional_fee';
	const ITEM_MISCELLANEOUS_FEE 		= 'item_miscellaneous_fee';
	const ITEM_PRICE 					= 'item_price';
	const ITEM_DISCOUNT 				= 'item_discount';
	const ITEM_TAX_TOTAL 				= 'item_tax_total';
	const ITEM_SUBTOTAL 				= 'item_subtotal';
	const ITEM_EXPENSE_TOTAL 			= 'item_expense_total';
	const ITEM_TOTAL 					= 'item_total';
	const TABLE_NAME 					= 'invoice_items';
	protected $table 					= self::TABLE_NAME;
	public $incrementing 				= true;
	protected $primaryKey 				= self::INVOICE_ITEM_ID;
	public $timestamps 					= false;

	protected $casts = [
		self::INVOICE_ITEM_ID 			=> 'int',
		self::INVOICE_ID 				=> 'int',
		self::ITEM_ID 					=> 'int',
		self::PROVINCE_SERVICE_ID 		=> 'int',
		self::ITEM_DATE_ADDED 			=> 'datetime',
		self::ITEM_QUANTITY 			=> 'float',
		self::ITEM_PROFESSIONAL_FEE 	=> 'float',
		self::ITEM_MISCELLANEOUS_FEE 	=> 'float',
		self::ITEM_PRICE 				=> 'float',
		self::ITEM_DISCOUNT 			=> 'float',
		self::ITEM_TAX_TOTAL 			=> 'float',
		self::ITEM_SUBTOTAL 			=> 'float',
		self::ITEM_EXPENSE_TOTAL 		=> 'float',
		self::ITEM_TOTAL 				=> 'float'
	];

	protected $fillable = [
		self::INVOICE_ID,
		self::ITEM_ID,
		self::PROVINCE_SERVICE_ID,
		self::ITEM_DATE_ADDED,
		self::ITEM_NAME,
		self::ITEM_DESCRIPTION,
		self::ITEM_QUANTITY,
		self::ITEM_PROFESSIONAL_FEE,
		self::ITEM_MISCELLANEOUS_FEE,
		self::ITEM_PRICE,
		self::ITEM_DISCOUNT,
		self::ITEM_TAX_TOTAL,
		self::ITEM_SUBTOTAL,
		self::ITEM_EXPENSE_TOTAL,
		self::ITEM_TOTAL
	];

	protected $attributes = [
		self::ITEM_PROFESSIONAL_FEE		=> 0.0,
		self::ITEM_MISCELLANEOUS_FEE	=> 0.0,
		self::ITEM_DISCOUNT				=> 0.0,
		self::ITEM_SUBTOTAL				=> 0.0,
		self::ITEM_EXPENSE_TOTAL		=> 0.0,
		self::ITEM_TAX_TOTAL			=> 0.0,
		self::ITEM_TOTAL				=> 0.0,
	];

	protected static array $lowercase = [
		self::ITEM_NAME,
		self::ITEM_DESCRIPTION	
	];

	/**
	 * Returns the invoice this invoice item belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function invoice(): BelongsTo
	{
		return $this->belongsTo(Invoice::class, Invoice::INVOICE_ID);
	}

	/**
	 * Returns the item of this invoice item
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function item(): BelongsTo
	{
		return $this->belongsTo(Item::class, Item::ITEM_ID);
	}

	/**
	 * Returns the province service this item belongs to - copied from item
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function provinceService(): BelongsTo
	{
		return $this->belongsTo(ProvinceService::class, ProvinceService::PROVINCE_SERVICE_ID);
	}

	/**
	 * Returns the expenses attached to this invoice item
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function expenses(): HasMany
	{
		return $this->hasMany(InvoiceItemExpense::class, self::INVOICE_ITEM_ID);
	}

	public static function factory(): InvoiceItemFactory
	{
		return InvoiceItemFactory::new();
	}
}
