<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Invoices;

use App\Observers\Invoices\InvoiceObserver;
use Carbon\Carbon;
use Database\Factories\Invoices\InvoiceFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

//Models
use App\Models\Quotes\Quote;
use App\Models\Items\BusinessItem;
use App\Models\Invoices\InvoiceItem;
use App\Models\Items\Item;
use App\Models\Invoices\InvoiceTerms;

//Enums
use App\Enums\Invoices\InvoiceStatus;

//Traits
use App\Traits\Businesses\BelongsToBusiness;
use App\Traits\Locations\BelongsToFranchise;
use App\Traits\Users\BelongsToUser;
use App\Traits\Users\Notable;
use App\Traits\Finances\Payable;
use App\Traits\Users\Uploadable;

/**
 * Class Invoice
 * 
 * @property int $invoice_id
 * @property int $user_id
 * @property int $business_id
 * @property int $franchise_id
 * @property int|null $quote_id
 * @property int|null $invoice_terms_id
 * @property string $invoice_number
 * @property InvoiceStatus::class $invoice_status
 * @property string|null $invoice_password
 * @property string|null $invoice_url_key
 * @property bool $is_read_only
 * @property bool $invoice_is_altered
 * @property float|null $invoice_item_discount_total
 * @property float|null $invoice_item_tax_total
 * @property float|null $invoice_item_subtotal
 * @property float|null $invoice_balance
 * @property float|null $invoice_paid
 * @property Carbon|null $invoice_date_due
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property string|null $deleted_at
 * 
 * @property Quote|null $quote
 * @property Collection|BusinessItem[] $businessItems
 * @property Collection|Item[] $items
 * @property Collection|InvoiceItem[] $invoiceItems
 * @property InvoiceTerms|null $terms
 *
 * @package App\Models
 */

#[ObservedBy(InvoiceObserver::class)]
class Invoice extends Model
{
	use SoftDeletes;
	use BelongsToBusiness;
	use BelongsToUser;
	use Payable;
	use Uploadable;
	use Notable;
	use HasFactory;
	use BelongsToFranchise; //A quote will belong to a franchise, it will be the franchise the user belongs to, the franchise they made it from

	const INVOICE_ID 						= 'invoice_id';
	const USER_ID 							= 'user_id';
	const BUSINESS_ID 						= 'business_id';
	const FRANCHISE_ID 						= 'franchise_id';
	const QUOTE_ID 							= 'quote_id';
	const INVOICE_TERMS_ID 					= 'invoice_terms_id';
	const INVOICE_NUMBER 					= 'invoice_number';
	const INVOICE_STATUS 					= 'invoice_status';
	const INVOICE_PASSWORD 					= 'invoice_password';
	const INVOICE_URL_KEY 					= 'invoice_url_key';
	const IS_READ_ONLY 						= 'is_read_only';
	const INVOICE_IS_ALTERED 				= 'invoice_is_altered';
	const INVOICE_ITEM_DISCOUNT_TOTAL 		= 'invoice_item_discount_total';
	const INVOICE_ITEM_TAX_TOTAL 			= 'invoice_item_tax_total';
	const INVOICE_ITEM_SUBTOTAL 			= 'invoice_item_subtotal';
	const INVOICE_BALANCE 					= 'invoice_balance';
	const INVOICE_PAID 						= 'invoice_paid';
	const INVOICE_DATE_DUE 					= 'invoice_date_due';
	const CREATED_AT 						= 'created_at';
	const UPDATED_AT 						= 'updated_at';
	const DELETED_AT 						= 'deleted_at';
	const TABLE_NAME 						= 'invoices';
	protected $table 						= self::TABLE_NAME;
	protected $primaryKey 					= self::INVOICE_ID;
	public $incrementing 					= true;

	protected $casts = [
		self::INVOICE_ID 					=> 'int',
		self::USER_ID 						=> 'int',
		self::BUSINESS_ID 					=> 'int',
		self::FRANCHISE_ID 					=> 'int',
		self::QUOTE_ID 						=> 'int',
		self::INVOICE_TERMS_ID 				=> 'int',
		self::INVOICE_STATUS 				=> InvoiceStatus::class,
		self::IS_READ_ONLY		 			=> 'bool',
		self::INVOICE_IS_ALTERED 			=> 'bool',
		self::INVOICE_ITEM_DISCOUNT_TOTAL 	=> 'float',
		self::INVOICE_ITEM_TAX_TOTAL 		=> 'float',
		self::INVOICE_ITEM_SUBTOTAL 		=> 'float',
		self::INVOICE_BALANCE 				=> 'float',
		self::INVOICE_PAID 					=> 'float',
		self::INVOICE_DATE_DUE 				=> 'datetime',
		self::CREATED_AT 					=> 'datetime',
		self::UPDATED_AT 					=> 'datetime'
	];

	protected $hidden = [
		self::INVOICE_PASSWORD
	];

	protected $fillable = [
		self::USER_ID,
		self::BUSINESS_ID,
		self::FRANCHISE_ID,
		self::QUOTE_ID,
		self::INVOICE_TERMS_ID,
		self::INVOICE_NUMBER,
		self::INVOICE_STATUS,
		self::INVOICE_PASSWORD,
		self::INVOICE_URL_KEY,
		self::IS_READ_ONLY,
		self::INVOICE_IS_ALTERED,
		self::INVOICE_ITEM_DISCOUNT_TOTAL,
		self::INVOICE_ITEM_TAX_TOTAL,
		self::INVOICE_ITEM_SUBTOTAL,
		self::INVOICE_BALANCE,
		self::INVOICE_PAID,
		self::INVOICE_DATE_DUE
	];

	protected $attributes = [
		self::IS_READ_ONLY					=> false,
		self::INVOICE_IS_ALTERED			=> false,
		self::INVOICE_ITEM_DISCOUNT_TOTAL	=> 0.0,
		self::INVOICE_ITEM_TAX_TOTAL		=> 0.0,
		self::INVOICE_ITEM_SUBTOTAL			=> 0.0,
		self::INVOICE_BALANCE				=> 0.0,
		self::INVOICE_PAID					=> 0.0,
	];

	/**
	 * Returns the quote that this invoice was generated from
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function quote(): BelongsTo
	{
		return $this->belongsTo(Quote::class, Quote::QUOTE_ID);
	}

	/**
	 * Returns the Business Items that were created from this invoice
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function businessItems(): HasMany
	{
		return $this->hasMany(BusinessItem::class, self::INVOICE_ID);
	}

	/**
	 * Returns the items that this invoice has
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function items(): BelongsToMany
	{
		return $this->belongsToMany(Item::class, 
									InvoiceItem::INVOICE_ITEM_ID, 
									self::INVOICE_ID, 
									Item::ITEM_ID)
					->withPivot(InvoiceItem::INVOICE_ITEM_ID, InvoiceItem::PROVINCE_SERVICE_ID, 
								InvoiceItem::ITEM_DATE_ADDED, InvoiceItem::ITEM_NAME, 
								InvoiceItem::ITEM_DESCRIPTION, InvoiceItem::ITEM_QUANTITY, 
								InvoiceItem::ITEM_PROFESSIONAL_FEE, InvoiceItem::ITEM_MISCELLANEOUS_FEE, 
								InvoiceItem::ITEM_PRICE, InvoiceItem::ITEM_DISCOUNT, 
								InvoiceItem::ITEM_TAX_TOTAL, InvoiceItem::ITEM_SUBTOTAL, 
								InvoiceItem::ITEM_TOTAL);
	}

	/**
	 * Returns the invoice items attached to this invoice
	 * - This would be used if the item is deleted, we soft copied the item info into invoice items
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function invoiceItems()
	{
		return $this->hasMany(InvoiceItem::class, self::INVOICE_ID);
	}

	/**
	 * Returns the terms this invoice has
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function terms() {
		return $this->belongsTo(InvoiceTerms::class, InvoiceTerms::INVOICE_TERMS_ID);
	}

	public static function factory(): InvoiceFactory
	{
		return InvoiceFactory::new();
	}
}
