<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Finances;

use Carbon\Carbon;
use Database\Factories\Finances\ExpenseFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

//Models
use App\Models\Finances\Creditor;
use App\Models\Finances\ExpenseCategory;

//Enums
use App\Enums\Finances\ExpenseStatus;
use App\Enums\Finances\ExpenseType;

//Traits
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;
use App\Traits\Finances\Payable;

/**
 * Class Expense
 * 
 * @property int $expense_id
 * @property int|null $creditor_id
 * @property int|null $expense_category_id
 * @property string $expense_description
 * @property float $expense_total
 * @property Carbon $expense_date
 * @property float $expense_paid
 * @property ExpenseType::class $expense_type
 * @property ExpenseStatus::class $expense_status
 * 
 * @property Creditor|null $creditor
 * @property ExpenseCategory|null $expenseCategory
 *
 * @package App\Models
 */
class Expense extends Model
{
	use Payable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const EXPENSE_ID 				= 'expense_id';
	const CREDITOR_ID 				= 'creditor_id';
	const EXPENSE_CATEGORY_ID 		= 'expense_category_id';
	const EXPENSE_DESCRIPTION 		= 'expense_description';
	const EXPENSE_TOTAL 			= 'expense_total';
	const EXPENSE_DATE 				= 'expense_date';
	const EXPENSE_PAID 				= 'expense_paid';
	const EXPENSE_TYPE 				= 'expense_type';
	const EXPENSE_STATUS 			= 'expense_status';
	const TABLE_NAME 				= 'expenses';
	protected $table 				= self::TABLE_NAME;
	protected $primaryKey 			= self::EXPENSE_ID;
	public $incrementing 			= true;
	public $timestamps 				= false;

	protected $casts = [
		self::EXPENSE_ID 			=> 'int',
		self::CREDITOR_ID 			=> 'int',
		self::EXPENSE_CATEGORY_ID 	=> 'int',
		self::EXPENSE_TOTAL 		=> 'float',
		self::EXPENSE_DATE 			=> 'datetime',
		self::EXPENSE_PAID 			=> 'float',
		self::EXPENSE_TYPE 			=> ExpenseType::class,
		self::EXPENSE_STATUS 		=> ExpenseStatus::class
	];

	protected $fillable = [
		self::CREDITOR_ID,
		self::EXPENSE_CATEGORY_ID,
		self::EXPENSE_DESCRIPTION,
		self::EXPENSE_TOTAL,
		self::EXPENSE_DATE,
		self::EXPENSE_PAID,
		self::EXPENSE_TYPE,
		self::EXPENSE_STATUS
	];

	protected static array $lowercase = [
		self::EXPENSE_DESCRIPTION	
	];

	/**
	 * Returns the creditor that created this expense
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function creditor(): BelongsTo
	{
		return $this->belongsTo(Creditor::class, Creditor::CREDITOR_ID);
	}

	/**
	 * Returns the expense category this expense belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function expenseCategory(): BelongsTo
	{
		return $this->belongsTo(ExpenseCategory::class, ExpenseCategory::EXPENSE_CATEGORY_ID);
	}

	public static function factory(): ExpenseFactory
	{
		return ExpenseFactory::new();
	}
}
