<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Finances;

use Database\Factories\Finances\BankAccountFactory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Finances\Bank;
use App\Models\Locations\Province;
use App\Models\Users\User;

//Enums
use App\Enums\Finances\BankAccountType;

//Traits
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;

/**
 * Class BankAccount
 * 
 * @property int $bank_account_id
 * @property int $bank_id
 * @property string $account_holder
 * @property string $account_number
 * @property BankAccountType::class $account_type
 * @property string $branch_code
 * 
 * @property Bank $bank
 * @property Collection|Province[] $provinces
 * @property Collection|User[] $users
 *
 * @package App\Models
 */
class BankAccount extends Model
{
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const BANK_ACCOUNT_ID 		= 'bank_account_id';
	const BANK_ID 				= 'bank_id';
	const ACCOUNT_HOLDER 		= 'account_holder';
	const ACCOUNT_NUMBER 		= 'account_number';
	const ACCOUNT_TYPE 			= 'account_type';
	const BRANCH_CODE 			= 'branch_code';
	const TABLE_NAME 			= 'bank_accounts';
	protected $table 			= self::TABLE_NAME;
	protected $primaryKey 		= self::BANK_ACCOUNT_ID;
	public $incrementing 		= true;
	public $timestamps 			= false;

	protected $casts = [
		self::BANK_ACCOUNT_ID 	=> 'int',
		self::BANK_ID 			=> 'int',
		self::ACCOUNT_TYPE 		=> BankAccountType::class
	];

	protected $fillable = [
		self::BANK_ID,
		self::ACCOUNT_HOLDER,
		self::ACCOUNT_NUMBER,
		self::ACCOUNT_TYPE,
		self::BRANCH_CODE
	];

	protected static array $lowercase = [
		self::ACCOUNT_HOLDER
	]; 

	/**
	 * Returns the bank this bank account belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function bank(): BelongsTo
	{
		return $this->belongsTo(Bank::class, Bank::BANK_ID);
	}

	/**
	 * Returns the provinces that use this bank account
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function provinces(): HasMany
	{
		return $this->hasMany(Province::class, self::BANK_ACCOUNT_ID);
	}

	/**
	 * Retruns the users that use this bank account
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function users(): HasMany
	{
		return $this->hasMany(User::class, self::BANK_ACCOUNT_ID);
	}

	public function getFullBankAccountAttribute() {
		return implode(', ', [
			$this->account_holder,
			$this->account_number,
			$this->account_type->value,
			$this->branch_code,
		]);
	}

	public static function factory(): BankAccountFactory
	{
		return BankAccountFactory::new();
	}
}
