<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Contacts;

use App\Observers\Contacts\ContactObserver;
use Database\Factories\Contacts\ContactFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

//Models
use App\Models\Users\Client;
use App\Models\Contacts\ContactMethod;
use App\Models\Users\Official;
use App\Models\Users\User;

//Enums
use App\Enums\Contacts\GenderType;

//Traits
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Contacts\ManyAddressable;
use App\Traits\Users\Notable;

/**
 * Class Contact
 * 
 * @property int $contact_id
 * @property string $contact_first_name
 * @property string $contact_last_name
 * @property string|null $contact_id_number
 * @property string|null $contact_passport_number
 * @property GenderType::class $gender
 * 
 * @property Client|null $client
 * @property Collection|ContactMethod[] $contactMethods
 * @property Official|null $official
 * @property User|null $user
 *
 * @package App\Models
 */

#[ObservedBy(ContactObserver::class)]
class Contact extends Model
{
	use ManyAddressable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const CONTACT_ID 					= 'contact_id';
	const CONTACT_FIRST_NAME 			= 'contact_first_name';
	const CONTACT_LAST_NAME 			= 'contact_last_name';
	const CONTACT_ID_NUMBER 			= 'contact_id_number';
	const CONTACT_PASSPORT_NUMBER 		= 'contact_passport_number';
	const CONTACT_GENDER 				= 'contact_gender';
	const TABLE_NAME 					= 'contacts';
	protected $table 					= self::TABLE_NAME;
	protected $primaryKey 				= self::CONTACT_ID;
	public $incrementing 				= true;
	public $timestamps 					= false;

	protected $casts = [
		self::CONTACT_ID 				=> 'int',
		self::CONTACT_GENDER 			=> GenderType::class
	];

	protected $fillable = [
		self::CONTACT_FIRST_NAME,
		self::CONTACT_LAST_NAME,
		self::CONTACT_ID_NUMBER,
		self::CONTACT_PASSPORT_NUMBER,
		self::CONTACT_GENDER
	];

	protected $attributes = [
		self::CONTACT_GENDER			=> GenderType::OTHER,
	];

	protected static array $lowercase = [
		self::CONTACT_FIRST_NAME,
		self::CONTACT_LAST_NAME	
	];

	/**
	 * Returns the client this contact is associated with
	 * @return \Illuminate\Database\Eloquent\Relations\HasOne
	 */
	public function client(): HasOne
	{
		return $this->hasOne(Client::class, self::CONTACT_ID);
	}

	/**
	 * Returns the contact methods this contact has
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function contactMethods(): HasMany
	{
		return $this->hasMany(ContactMethod::class, self::CONTACT_ID);
	}

	/**
	 * Returns the official associated with this contact
	 * @return \Illuminate\Database\Eloquent\Relations\HasOne
	 */
	public function official(): HasOne
	{
		return $this->hasOne(Official::class, self::CONTACT_ID);
	}

	/**
	 * returns the user associated with this contact 
	 * @return \Illuminate\Database\Eloquent\Relations\HasOne
	 */
	public function user(): HasOne
	{
		return $this->hasOne(User::class, self::CONTACT_ID);
	}

	public function getFullNameAttribute() {
		return implode(', ', [
			$this->contact_first_name,
			$this->contact_last_name,
		]);
	}

	protected static function newFactory(): ContactFactory
    {
        return ContactFactory::new();
    }
}
