<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Contacts;

use Database\Factories\Contacts\AddressableFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

//Models
use App\Models\Businesses\Business;
use App\Models\Contacts\Contact;
use App\Models\Locations\LiquorAuthority;
use App\Models\Locations\Department;
use App\Models\Locations\Franchise;

//Enums
use App\Enums\Locations\AddressRelationship;
use App\Enums\Locations\AddressableType;

//Traits
use App\Traits\Locations\BelongsToAddress;
use App\Traits\Users\Notable;

/**
 * Class Addressable
 * 
 * @property int $entity_address_id
 * @property int $address_id
 * @property AddressableType::class $addressable_type
 * @property int $addressable_id
 * @property AddressRelationship::class $address_relationship
 * 
 * @property \Illuminate\Database\Eloquent\Model|Business|Contact|LiquorAuthority|Department|Franchise $addressable
 * @property-read Business|null $business
 * @property-read Contact|null $contact
 * @property-read LiquorAuthority|null $liquorAuthority
 * @property-read Department|null $department
 * @property-read Franchise|null $franchise
 *
 * @package App\Models
 */
class Addressable extends Model
{
	use BelongsToAddress;
	use Notable;
	use HasFactory;

	const ENTITY_ADDRESS_ID 		= 'entity_address_id';
	const ADDRESS_ID 				= 'address_id';
	const ADDRESSABLE_TYPE 			= 'addressable_type';
	const ADDRESSABLE_ID 			= 'addressable_id';
	const ADDRESS_RELATIONSHIP 		= 'address_relationship';
	const TABLE_NAME 				= 'addressables';
	const MORPH_NAME 				= 'addressable';
	protected $table 				= self::TABLE_NAME;
	protected $primaryKey 			= self::ENTITY_ADDRESS_ID;
	public $incrementing 			= true;
	public $timestamps 				= false;

	protected $casts = [
		self::ENTITY_ADDRESS_ID 	=> 'int',
		self::ADDRESS_ID 			=> 'int',
		self::ADDRESSABLE_TYPE 		=> AddressableType::class,
		self::ADDRESSABLE_ID		=> 'int',
		self::ADDRESS_RELATIONSHIP 	=> AddressRelationship::class
	];

	protected $fillable = [
		self::ADDRESS_ID,
		self::ADDRESSABLE_TYPE,
		self::ADDRESSABLE_ID,
		self::ADDRESS_RELATIONSHIP
	];

	/**
	 * Get the model (business, user, etc.) that this addressable belongs to.
	 */
	public function addressable()
	{
		return $this->morphTo();
	}

	/**
	 * Returns the business this addressable belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function business(): ?Business
	{
		return $this->addressable_type === Business::class
			? $this->addressable()
			: null;
	}

	/**
	 * Returns the contact this addressable belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function contact(): ?Contact
	{
		return $this->addressable_type === Contact::class
			? $this->addressable()
			: null;
	}

	/**
	 * Returns the liquorAuthority this addressable belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function liquorAuthority(): ?LiquorAuthority
	{
		return $this->addressable_type === LiquorAuthority::class
			? $this->addressable()
			: null;
	}

	/**
	 * Returns the department this addressable belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function department(): ?Department
	{
		return $this->addressable_type === Department::class
			? $this->addressable()
			: null;
	}

	/**
	 * Returns the franchise this addressable belongs to
	 * @return \Illuminate\Database\Eloquent\Relations\MorphTo|null
	 */
	public function franchise(): ?Franchise
	{
		return $this->addressable_type === Franchise::class
			? $this->addressable()
			: null;
	}

	public static function factory(): AddressableFactory
	{
		return AddressableFactory::new();
	}
}
