<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Contacts;

use Database\Factories\Contacts\AddressFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

//Models
use App\Models\Contacts\AddressExtraField;
use App\Models\Contacts\Addressable;
use App\Models\Contacts\Contact;
use App\Models\Businesses\Business;
use App\Models\Locations\LiquorAuthority;
use App\Models\Locations\Department;
use App\Models\Locations\Franchise;

//Enums
use App\Enums\Locations\ProvinceType;

//Traits
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Users\Notable;

/**
 * Class Address
 * 
 * @property int $address_id
 * @property string $street_address
 * @property string $suburb
 * @property string $city
 * @property ProvinceType $province
 * @property string $postal_code
 * 
 * @property \Illuminate\Database\Eloquent\Collection|AddressExtraField[] $extraFields
 * @property \Illuminate\Database\Eloquent\Collection|Addressable[] $addressables
 * @property \Illuminate\Database\Eloquent\Collection|Contact[] $contacts
 * @property \Illuminate\Database\Eloquent\Collection|Business[] $businesses
 * @property \Illuminate\Database\Eloquent\Collection|LiquorAuthority[] $liquorAuthorities
 * @property \Illuminate\Database\Eloquent\Collection|Department[] $departments
 * @property \Illuminate\Database\Eloquent\Collection|Franchise[] $franchises
 *
 * @package App\Models
 */
class Address extends Model
{
	use Notable;
	use HasFactory;
	use LowercaseAttributes;
	
	const ADDRESS_ID 		= 'address_id';
	const STREET_ADDRESS 	= 'street_address';
	const SUBURB 			= 'suburb';
	const CITY 				= 'city';
	const PROVINCE 			= 'province';
	const POSTAL_CODE 		= 'postal_code';
	const TABLE_NAME 		= 'addresses';
	protected $table 		= self::TABLE_NAME;
	protected $primaryKey 	= self::ADDRESS_ID;
	public $incrementing 	= true;
	public $timestamps 		= false;

	protected $casts = [
		self::ADDRESS_ID 	=> 'int',
		self::PROVINCE		=> ProvinceType::class,
	];

	protected $fillable = [
		self::STREET_ADDRESS,
		self::SUBURB,
		self::CITY,
		self::PROVINCE,
		self::POSTAL_CODE
	];

	protected static array $lowercase = [
		self::STREET_ADDRESS,
		self::SUBURB,
		self::CITY,
	];
	
	/**
	 * Returns the fields associated with this address
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function extraFields(): HasMany
	{
		return $this->hasMany(AddressExtraField::class, self::ADDRESS_ID);
	}

	/**
	 * Returns all the addressables associated with this address
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function addressables(): HasMany
	{
		return $this->hasMany(Addressable::class, self::ADDRESS_ID);
	}

	/**
	 * Returns the contacts that are linked to this address
	 * @return \Illuminate\Database\Eloquent\Relations\MorphToMany
	 */
	public function contacts()
	{
		return $this->morphedByMany(contact::class, 
									Addressable::MORPH_NAME, 
									Addressable::TABLE_NAME, 
									self::ADDRESS_ID, 
									Addressable::ADDRESSABLE_ID)
                    ->withPivot(Addressable::ADDRESS_RELATIONSHIP);
	}

	/**
	 * Returns the businesses that are associated with this address
	 * @return \Illuminate\Database\Eloquent\Relations\MorphToMany
	 */
	public function businesses()
	{
		return $this->morphedByMany(Business::class, 
									Addressable::MORPH_NAME, 
									Addressable::TABLE_NAME, 
									self::ADDRESS_ID, 
									Addressable::ADDRESSABLE_ID)
                    ->withPivot(Addressable::ADDRESS_RELATIONSHIP);
	}

	/**
	 * Returns the liquor authorities associated with this address
	 * @return \Illuminate\Database\Eloquent\Relations\MorphToMany
	 */
	public function liquorAuthorities()
	{
		return $this->morphedByMany(LiquorAuthority::class, 
									Addressable::MORPH_NAME, 
									Addressable::TABLE_NAME, 
									self::ADDRESS_ID, 
									Addressable::ADDRESSABLE_ID)
                    ->withPivot(Addressable::ADDRESS_RELATIONSHIP);
	}

	/**
	 * Return the departments that are associated with this adress
	 * @return \Illuminate\Database\Eloquent\Relations\MorphToMany
	 */
	public function departments()
	{
		return $this->morphedByMany(Department::class, 
									Addressable::MORPH_NAME, 
									Addressable::TABLE_NAME, 
									self::ADDRESS_ID, 
									Addressable::ADDRESSABLE_ID)
					->withPivot(Addressable::ADDRESS_RELATIONSHIP);
	}

	/**
	 * Return the franchises that are associated with this address
	 * @return \Illuminate\Database\Eloquent\Relations\MorphToMany
	 */
	public function franchises()
	{
		return $this->morphedByMany(Franchise::class, 
									Addressable::MORPH_NAME, 
									Addressable::TABLE_NAME, 
									self::ADDRESS_ID, 
									Addressable::ADDRESSABLE_ID)
					->withPivot(Addressable::ADDRESS_RELATIONSHIP);
	}

	public function getFullAddressAttribute() {
		return implode(', ', [
			$this->street_address,
			$this->suburb,
			$this->city,
			$this->province->value,
		]);
	}

	public static function factory(): AddressFactory
	{
		return AddressFactory::new();
	}
}
