<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models\Businesses;

use App\Observers\Businesses\BusinessObserver;
use Carbon\Carbon;
use Database\Factories\Businesses\BusinessFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

//Models
use App\Models\Users\Client;
use App\Models\Items\Item;
use App\Models\Invoices\Invoice;
use App\Models\Quotes\Quote;
use App\models\Businesses\BusinessClient;
use App\Models\Items\BusinessItem;

//Enums
use App\Enums\Businesses\BusinessType;

//Traits
use App\Traits\Users\BelongsToUser;
use App\Traits\Users\Changable;
use App\Traits\Core\LowercaseAttributes;
use App\Traits\Contacts\ManyAddressable;
use App\Traits\Users\Notable;
use App\Traits\Users\Uploadable;

/**
 * Class Business
 * 
 * @property int $business_id
 * @property int|null $user_id
 * @property string|null $business_reg_number
 * @property string $business_reg_name
 * @property string $business_trading_name
 * @property string|null $business_VAT_number
 * @property BusinessType::class $business_type
 * @property Carbon|null $business_reg_date
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property string|null $deleted_at
 * 
 * @property Collection|Client[] $clients
 * @property Collection|Item[] $items
 * @property Collection|Invoice[] $invoices
 * @property Collection|Quote[] $quotes
 *
 * @package App\Models
 */

#[ObservedBy(BusinessObserver::class)]
class Business extends Model
{
	use SoftDeletes;
	use BelongsToUser;
	use Changable;
	use ManyAddressable;
	use Uploadable;
	use Notable;
	use HasFactory;
	use LowercaseAttributes;

	const BUSINESS_ID 				= 'business_id';
	const USER_ID 					= 'user_id';
	const BUSINESS_REG_NUMBER 		= 'business_reg_number';
	const BUSINESS_REG_NAME 		= 'business_reg_name';
	const BUSINESS_TRADING_NAME 	= 'business_trading_name';
	const BUSINESS_VAT_NUMBER 		= 'business_VAT_number';
	const BUSINESS_TYPE 			= 'business_type';
	const BUSINESS_REG_DATE 		= 'business_reg_date';
	const CREATED_AT 				= 'created_at';
	const UPDATED_AT 				= 'updated_at';
	const DELETED_AT 				= 'deleted_at';
	const TABLE_NAME 				= 'businesses';
	protected $table 				= self::TABLE_NAME;
	public $incrementing 			= true;
	protected $primaryKey 			= self::BUSINESS_ID;

	protected $casts = [
		self::BUSINESS_ID 			=> 'int',
		self::USER_ID 				=> 'int',
		self::BUSINESS_TYPE 		=> BusinessType::class,
		self::BUSINESS_REG_DATE 	=> 'datetime',
		self::CREATED_AT 			=> 'datetime',
		self::UPDATED_AT 			=> 'datetime'
	];

	protected $fillable = [
		self::USER_ID,
		self::BUSINESS_REG_NUMBER,
		self::BUSINESS_REG_NAME,
		self::BUSINESS_TRADING_NAME,
		self::BUSINESS_VAT_NUMBER,
		self::BUSINESS_TYPE,
		self::BUSINESS_REG_DATE
	];

	protected static array $lowercase = [
		self::BUSINESS_REG_NAME,
		self::BUSINESS_TRADING_NAME,	
	];

	/**
	 * Returns the clients associated with this business
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function clients(): BelongsToMany
	{
		return $this->belongsToMany(Client::class, 
									BusinessClient::TABLE_NAME, 
									self::BUSINESS_ID, 
									Client::CLIENT_ID)
					->withPivot(BusinessClient::BUSINESS_CLIENT_ID, BusinessClient::CLIENT_ROLE, BusinessClient::IS_PRIMARY);
	}

	/**
	 * Returns the items that belong to this business
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function items(): BelongsToMany
	{
		return $this->belongsToMany(Item::class, 
									BusinessItem::TABLE_NAME, 
									self::BUSINESS_ID, 
									Item::ITEM_ID)
					->withPivot(BusinessItem::BUSINESS_ITEM_ID, BusinessItem::INVOICE_ID, BusinessItem::ITEM_STEP_ID, 
								BusinessItem::APPROVED_DATE, BusinessItem::RENEW_DATE, BusinessItem::EXPIRY_DATE, 
								BusinessItem::ITEM_STATUS, BusinessItem::ITEM_NAME, BusinessItem::ITEM_DESCRIPTION, 
								BusinessItem::ITEM_PRICE)
					->withTimestamps();
	}

	/**
	 * Returns the invoices belonging to this business
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function invoices(): HasMany
	{
		return $this->hasMany(Invoice::class, self::BUSINESS_ID);
	}

	/**
	 * Returns the quotes that belong to this business
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function quotes(): HasMany
	{
		return $this->hasMany(Quote::class, self::BUSINESS_ID);
	}

	#TODO : Add a scope feature to get quotes/invoices that are: paid, overdue etc

	public static function factory(): BusinessFactory
	{
		return BusinessFactory::new();
	}
}
